# -*- coding: utf-8 -*-
import os
import re
import time
import json
import threading
import requests
import urllib3
import tkinter as tk
from tkinter import filedialog, messagebox, ttk, scrolledtext, font
from datetime import datetime

# =============================================================================
# ⭐ 基础配置 URL = "xxxxxx" maybe add /v1/chat/completions
# =============================================================================
urllib3.disable_warnings(urllib3.exceptions.InsecureRequestWarning)

DEFAULT_API_URL = "http://api.example.com/v1/chat/completions"
DEFAULT_API_KEY = "sk-xxxxx"
DEFAULT_MODEL_NAME = "xxxxx"
DEFAULT_MAX_TOKENS = 800

# =============================================================================
# 🌍 国际化资源字典 (I18N) - 12 Language Support
# =============================================================================

UI_STRINGS = {
    'zh': {
        'app_title': "佛典AI译丛 (BCAITS) 通用文献梳理系统 v1.0 (学术严谨版)",
        'group_api': "API 与 模型配置",
        'lbl_url': "API 地址:",
        'lbl_key': "API Key:",
        'lbl_model': "模型名称:",
        'group_file': "输入源 (支持经/律/论及注疏)",
        'btn_select_file': "📂 选择文件",
        'lbl_threshold': "分段阈值 (Tokens):",
        'group_lang': "任务矩阵 (自动生成：split + 独立文件 + 对照版)", 
        'col_lang': "语言",
        'col_trans': "严谨翻译",
        'col_exe': "科判与义理梳理", # 名称变更
        'group_log': "系统日志 (进度)",
        'group_preview': "实时预览 (内容)",
        'btn_start': "🚀 启动任务",
        'btn_stop': "🛑 停止",
        'msg_err_nofile': "错误：请先选择输入文件",
        'msg_err_notask': "错误：请至少勾选一项任务",
        'msg_stop': "⚠️ 正在停止...",
        'msg_read_start': "读取文本中...",
        'msg_seg_done': "✅ 智能分段完成：共 {} 个片段。",
        'msg_split_created': "已生成原始分段文件: {}",
        'msg_file_created': "创建文件: {}",
        'msg_processing': "处理片段 ({}/{}): {}",
        'msg_generating': "  -> [{}] 生成 {} ...",
        'msg_done_title': "完成",
        'msg_done_body': "处理完成！\n已生成 [split] 原文、独立文件及 [对照版] 文件。",
        'err_fatal': "❌ 错误: {}",
        'lang_zh': "中文", 'lang_en': "英文", 'lang_ja': "日文",
        'lang_ko': "韩文", 'lang_bo': "藏文", 'lang_pali': "巴利文",
        'lang_es': "西班牙语", 'lang_ru': "俄语", 'lang_fr': "法语", 'lang_de': "德语",
        'lang_hi': "印地语", 'lang_no': "挪威语",
    },
    'en': {
        'app_title': "BCAITS Buddhist Canon Scholarly System v1.0",
        'group_api': "API & Model Settings",
        'lbl_url': "API Endpoint:",
        'lbl_key': "API Key:",
        'lbl_model': "Model Name:",
        'group_file': "Input Source",
        'btn_select_file': "📂 Browse...",
        'lbl_threshold': "Threshold:",
        'group_lang': "Task Matrix",
        'col_lang': "Language",
        'col_trans': "Translation",
        'col_exe': "Structure & Analysis",
        'group_log': "Log",
        'group_preview': "Preview",
        'btn_start': "🚀 EXECUTE",
        'btn_stop': "🛑 STOP",
        'msg_err_nofile': "Error: Select file.",
        'msg_err_notask': "Error: Select at least one task.",
        'msg_stop': "⚠️ Stopping...",
        'msg_read_start': "Reading text...",
        'msg_seg_done': "✅ Segments: {}.",
        'msg_split_created': "Created split file: {}",
        'msg_file_created': "Created: {}",
        'msg_processing': "Processing ({}/{}): {}",
        'msg_generating': "  -> [{}] Generating {} ...",
        'msg_done_title': "Finished",
        'msg_done_body': "Done! Split file, Separate files and Contrast files generated.",
        'err_fatal': "❌ Error: {}",
        'lang_zh': "Chinese", 'lang_en': "English", 'lang_ja': "Japanese",
        'lang_ko': "Korean", 'lang_bo': "Tibetan", 'lang_pali': "Pali",
        'lang_es': "Spanish", 'lang_ru': "Russian", 'lang_fr': "French", 'lang_de': "German",
        'lang_hi': "Hindi", 'lang_no': "Norwegian",
    },
    'ja': {'app_title': "BCAITS v1.0 (学術版)", 'group_api':"API設定", 'lbl_url':"URL:", 'lbl_key':"Key:", 'lbl_model':"Model:", 'group_file':"入力", 'btn_select_file':"選択", 'lbl_threshold':"閾値:", 'group_lang':"タスク", 'col_lang':"言語", 'col_trans':"翻訳", 'col_exe':"科判・義理", 'group_log':"ログ", 'group_preview':"プレビュー", 'btn_start':"🚀 開始", 'btn_stop':"🛑 停止", 'msg_err_nofile':"ファイル未選択", 'msg_err_notask':"タスク未選択", 'msg_stop':"停止中...", 'msg_read_start':"読込中...", 'msg_seg_done':"完了: {}", 'msg_split_created':"分割ファイル: {}", 'msg_file_created':"作成: {}", 'msg_processing':"処理中 ({}/{}): {}", 'msg_generating':"-> [{}] 生成 {}", 'msg_done_title':"完了", 'msg_done_body':"完了", 'err_fatal':"エラー: {}", 'lang_zh':"中国語", 'lang_en':"英語", 'lang_ja':"日本語", 'lang_ko':"韓国語", 'lang_bo':"チベット語", 'lang_pali':"パーリ語", 'lang_es':"スペイン語", 'lang_ru':"ロシア語", 'lang_fr':"フランス語", 'lang_de':"ドイツ語", 'lang_hi':"ヒンディー語", 'lang_no':"ノルウェー語"},
}

# =============================================================================
# 🧠 AI 核心配置：学术梳理 (Scholarly Sorting)
# =============================================================================

class LangConfig:
    @staticmethod
    def get_trans_prompt(lang_code):
        # 翻译 Prompt：强调“信”（Accuracy）和学术严谨性
        base = "You are a professional philologist and translator of Buddhist texts."
        rules = "Translate the following text accurately, maintaining doctrinal consistency. Keep the original title. No commentary in the translation section."
        
        prompts = {
            'zh': f"{base} 请将文本翻译为准确、严谨的现代书面中文。{rules}",
            'en': f"{base} Translate into precise, academic English. {rules}",
            'ja': f"{base} 正確で学術的な日本語に翻訳してください。{rules}",
            'ko': f"{base} 정확하고 학술적인 한국어로 번역하십시오. {rules}",
            'bo': f"{base} Translate into Tibetan (Dharma language). {rules}",
            'pali': f"{base} Translate into Romanized Pali. {rules}",
            'es': f"{base} Translate into precise Spanish. {rules}",
            'ru': f"{base} Translate into precise Russian. {rules}",
            'fr': f"{base} Translate into precise French. {rules}",
            'de': f"{base} Translate into precise German. {rules}",
            'hi': f"{base} Translate into formal Hindi. {rules}",
            'no': f"{base} Translate into clear Norwegian. {rules}",
        }
        return prompts.get(lang_code, prompts['en'])

    @staticmethod
    def get_exe_prompt(lang_code):
        # 核心指令：降温、结构化、义理梳理
        CORE_INSTRUCTION = """
        [Role]: You are a rigorous Scholar of Buddhist Studies (Buddhologist). You focus on philology, structural logic, and doctrinal accuracy.
        [Task]: Do not be spontaneous or poetic. Instead, analyze the text logically and clearly.
        [Method]:
        1. **Structure (科判/Outline)**: Break down the text's logical flow (e.g., Introduction, Core Argument, Conclusion, or specific hierarchical divisions).
        2. **Terminology (名相/Philology)**: Explain key Buddhist terms, referencing Sanskrit/Pali roots if applicable.
        3. **Doctrine (义理/Analysis)**: Explain the meaning based on standard Buddhist philosophy (Abhidharma, Madhyamaka, Yogacara, etc.), avoiding subjective speculation.
        4. **Summary (大意)**: A concise summary of the passage.
        """
        
        prompts = {
            'zh': f"""{CORE_INSTRUCTION}
            输出格式：
            【科判结构】[分析文本的逻辑层次、段落大意]
            【名相释义】[解释关键法相名词，注明梵/巴原词]
            【义理阐释】[依据经论传统进行严谨解读，不做过度发挥]
            【本段大意】[平实概括]""",
            
            'en': f"""{CORE_INSTRUCTION}
            Output Format:
            [Structural Outline] (Logical breakdown)
            [Terminology] (Key terms with Etymology)
            [Doctrinal Analysis] (Rigorous explanation)
            [Summary]""",
            
            'ja': f"""{CORE_INSTRUCTION}
            出力形式：
            【科判・構造】
            【用語解説】
            【義理の解釈】(学術的解説)
            【概要】""",
            
            'ko': f"""{CORE_INSTRUCTION}
            출력 형식:
            【과판/구조】
            【명상 해석】
            【의리 해설】
            【요약】""",
            
            'bo': f"""{CORE_INSTRUCTION}
            Output Format:
            【Sa-bcad】 (Structure/Outline)
            【Tha-snyad】 (Terminology)
            【Gzhung-don】 (Doctrinal Meaning)
            【bsDus-don】 (Summary)""",
            
            'pali': f"""{CORE_INSTRUCTION}
            Output Format:
            【Nidāna & Pāḷi】
            【Padavanṇanā】 (Word Analysis)
            【Attha-Vannanā】 (Meaning)
            【Sāra】""",
            
            'es': f"""{CORE_INSTRUCTION} Output: [Estructura]; [Terminología]; [Análisis Doctrinal]; [Resumen].""",
            'ru': f"""{CORE_INSTRUCTION} Output: [Структура]; [Терминология]; [Доктринальный анализ]; [Резюме].""",
            'fr': f"""{CORE_INSTRUCTION} Output: [Structure]; [Terminologie]; [Analyse Doctrinale]; [Résumé].""",
            'de': f"""{CORE_INSTRUCTION} Output: [Struktur]; [Terminologie]; [Doktrinäre Analyse]; [Zusammenfassung].""",
            'hi': f"""{CORE_INSTRUCTION} Output (Hindi): [संरचना]; [शब्दावली]; [सिद्धांत विश्लेषण]; [सारांश].""",
            'no': f"""{CORE_INSTRUCTION} Output: [Struktur]; [Terminologi]; [Doktrinær Analyse]; [Sammendrag]."""
        }
        return prompts.get(lang_code, prompts['en'])

    @staticmethod
    def get_file_suffix(lang_code, task_type):
        file_lang_code = 'cn' if lang_code == 'zh' else lang_code
        
        # 命名映射：更学术化
        names = {
            'trans': {
                'zh': '严谨翻译', 'en': 'Translation', 'ja': '翻訳', 'ko': '번역', 'bo': 'Yig-sgyur', 'pali': 'Parivattana',
                'es': 'Traducción', 'ru': 'Perevod', 'fr': 'Traduction', 'de': 'Übersetzung', 'hi': 'Anuvad', 'no': 'Oversettelse'
            },
            'exe': {
                'zh': '义理梳理', 'en': 'Analysis', 'ja': '義理解説', 'ko': '의리해설', 'bo': 'rNam-bshad', 'pali': 'Vibhanga',
                'es': 'Análisis', 'ru': 'Analiz', 'fr': 'Analyse', 'de': 'Analyse', 'hi': 'Vishleshan', 'no': 'Analyse'
            },
            'combined': {
                'zh': '对照研读版', 'en': 'Study_Edition', 'ja': '対照版', 'ko': '대조판', 'bo': 'Shan-sbyar', 'pali': 'Samagga',
                'es': 'Estudio', 'ru': 'Izuchenie', 'fr': 'Étude', 'de': 'Studium', 'hi': 'Adhyayan', 'no': 'Studie'
            }
        }
        
        func_name = names.get(task_type, {}).get(lang_code, 'Output')
        return f"_{func_name}.{file_lang_code}.txt"

    @staticmethod
    def validate_pali(text):
        clean_text = re.sub(r'[（(][^)）]*[)）]', '', text.replace('\ufeff', ''))
        if re.search(r'[\u0900-\u097F]', clean_text): return False, "Detected Devanagari"
        return True, None

# =============================================================================
# 📜 智能分段 (保持不变，通用性强)
# =============================================================================

class ZenTextProcessor:
    def __init__(self):
        # 针对通用佛典，增加了“品、分、论”等关键词
        self.header_pattern = re.compile(r'^\s*([^\s]+(?:品|分|论|经|疏|章|节|卷第.+)|Chapter\s+\d+|Section\s+\d+)\s*$')
        self.sentence_end_pattern = re.compile(r'([。！？；.!?;།༎।]+)')

    def preprocess_text(self, full_text):
        if len(full_text) > 1000 and full_text.count('\n') < 10:
            full_text = self.sentence_end_pattern.sub(r'\1\n', full_text)
        return full_text

    def smart_segmentation(self, full_text, max_chars=3000):
        full_text = self.preprocess_text(full_text)
        lines = full_text.split('\n')
        segments = []
        current_title = "Excerpt / 选段"
        current_buffer = []
        current_count = 0
        is_standard_style = (sum(1 for line in lines[:100] if self.header_pattern.match(line)) > 2)

        for line in lines:
            line = line.rstrip()
            if not line: continue
            should_split = False; new_title = None
            is_header = False
            
            if is_standard_style:
                if self.header_pattern.match(line) and len(line) < 40: is_header = True
            else:
                if len(line) < 40 and not self.sentence_end_pattern.search(line[-1:]):
                    if current_count > 100: is_header = True
            
            if is_header: should_split = True; new_title = line.strip()
            if current_count + len(line) > max_chars: should_split = True; new_title = new_title if new_title else current_title + " (Part II)"

            if should_split and current_buffer:
                segments.append({"title": current_title, "content": "\n".join(current_buffer)})
                if new_title:
                    current_title = new_title
                    if is_header: current_buffer = []; current_count = 0
                    else: current_buffer = [line]; current_count = len(line)
                else: current_buffer = [line]; current_count = len(line)
            else:
                if is_header: current_title = line.strip()
                else: current_buffer.append(line); current_count += len(line)
        
        if current_buffer: segments.append({"title": current_title, "content": "\n".join(current_buffer)})
        if len(segments) > 1 and len(segments[-1]['content']) < (max_chars * 0.1):
            segments[-2]['content'] += "\n\n" + segments[-1]['content']; segments.pop()
        return segments

# =============================================================================
# 🤖 AI 引擎 (降温处理)
# =============================================================================

class AiEngine:
    def __init__(self, api_url, api_key):
        self.api_url = api_url
        self.api_key = api_key
    def process(self, title, content, system_prompt, model_name, validator=None):
        user_prompt = f"Target Text: {title}\n\nContent:\n{content}"
        messages = [{"role": "system", "content": system_prompt}, {"role": "user", "content": user_prompt}]
        headers = {"Authorization": f"Bearer {self.api_key}", "Content-Type": "application/json"}
        for _ in range(3):
            try:
                # ⭐ 关键修改：Temperature 设为 0.3，降低热度，追求严谨
                payload = {"model": model_name, "messages": messages, "temperature": 0.3, "max_tokens": 4096}
                resp = requests.post(self.api_url, headers=headers, json=payload, timeout=240)
                if resp.status_code!=200: raise Exception(str(resp.status_code))
                res = resp.json()['choices'][0]['message']['content']
                if validator:
                    if not validator(res)[0]: continue
                return res
            except: time.sleep(3)
        return "[FAILED]"

# =============================================================================
# 🚀 启动器
# =============================================================================

class LanguageLauncher:
    def __init__(self):
        self.root = tk.Tk(); self.root.title("Language Selection"); 
        w,h=550,550; x,y=(self.root.winfo_screenwidth()-w)//2, (self.root.winfo_screenheight()-h)//2
        self.root.geometry(f"{w}x{h}+{x}+{y}"); self.selected_lang=None
        ttk.Label(self.root, text="Select Interface Language\n请选择界面语言", font=("Arial",14), justify=tk.CENTER).pack(pady=20)
        f=ttk.Frame(self.root); f.pack(pady=10)
        
        langs=[
            ("中文 (Chinese)",'zh'),("English",'en'),
            ("日本語 (Japanese)",'ja'),("한국어 (Korean)",'ko'),
            ("བོད་ཡིག (Tibetan)",'bo'),("Pāḷi (Roman)",'pali'),
            ("Español (Spanish)",'es'),("Русский (Russian)",'ru'),
            ("Français (French)",'fr'),("Deutsch (German)",'de'),
            ("हिन्दी (Hindi)",'hi'),("Norsk (Norwegian)",'no')
        ]
        
        for i,(n,c) in enumerate(langs): 
            ttk.Button(f,text=n,command=lambda x=c:self.sel(x),width=22).grid(row=i//2,column=i%2,padx=10,pady=10)
    def sel(self,c): self.selected_lang=c; self.root.destroy()
    def run(self): self.root.mainloop(); return self.selected_lang

# =============================================================================
# 🖥️ GUI 主程序
# =============================================================================

class ZenUniversalApp:
    def __init__(self, root, ui_lang='zh'):
        self.root = root; self.ui_lang = ui_lang
        self.T = UI_STRINGS.get(ui_lang, UI_STRINGS['en'])
        self.root.title(self.T['app_title'])
        self.root.geometry("1200x950")
        self.processor = ZenTextProcessor()
        self.vars_trans = {}; self.vars_exe = {}
        self.is_running = False; self.stop_event = threading.Event()
        self._setup_ui()
        
    def _setup_ui(self):
        top_container = ttk.Frame(self.root, padding=10); top_container.pack(fill=tk.X)
        api_group = ttk.LabelFrame(top_container, text=self.T['group_api'], padding=10)
        api_group.pack(side=tk.LEFT, fill=tk.BOTH, expand=True)
        f1 = ttk.Frame(api_group); f1.pack(fill=tk.X, pady=2)
        ttk.Label(f1, text=self.T['lbl_url'], width=10).pack(side=tk.LEFT)
        self.api_url = tk.StringVar(value=DEFAULT_API_URL); ttk.Entry(f1, textvariable=self.api_url).pack(side=tk.LEFT, fill=tk.X, expand=True, padx=5)
        ttk.Label(f1, text=self.T['lbl_key'], width=8).pack(side=tk.LEFT)
        self.api_key = tk.StringVar(value=DEFAULT_API_KEY); ttk.Entry(f1, textvariable=self.api_key, show="*", width=20).pack(side=tk.LEFT, padx=5)
        f2 = ttk.Frame(api_group); f2.pack(fill=tk.X, pady=5)
        ttk.Label(f2, text=self.T['lbl_model'], width=10).pack(side=tk.LEFT)
        self.model_name = tk.StringVar(value=DEFAULT_MODEL_NAME); ttk.Entry(f2, textvariable=self.model_name).pack(side=tk.LEFT, fill=tk.X, expand=True, padx=5)
        
        action_frame = ttk.Frame(top_container, padding=(10, 0, 0, 0)); action_frame.pack(side=tk.RIGHT, fill=tk.Y)
        style = ttk.Style(); style.configure("Big.TButton", font=("Arial", 12, "bold")); style.configure("Stop.TButton", font=("Arial", 10))
        self.btn_start = ttk.Button(action_frame, text=self.T['btn_start'], command=self.start, style="Big.TButton", width=15)
        self.btn_start.pack(side=tk.TOP, fill=tk.BOTH, expand=True, pady=(5, 5), ipady=10)
        self.btn_stop = ttk.Button(action_frame, text=self.T['btn_stop'], command=self.stop, style="Stop.TButton")
        self.btn_stop.pack(side=tk.BOTTOM, fill=tk.X, pady=(0, 5))

        mid_frame = ttk.Frame(self.root, padding=10); mid_frame.pack(fill=tk.X)
        file_group = ttk.LabelFrame(mid_frame, text=self.T['group_file'], padding=10)
        file_group.pack(side=tk.LEFT, fill=tk.BOTH, expand=True, padx=(0, 5))
        f_box = ttk.Frame(file_group); f_box.pack(fill=tk.X)
        self.file_path = tk.StringVar(); ttk.Entry(f_box, textvariable=self.file_path).pack(side=tk.LEFT, fill=tk.X, expand=True)
        ttk.Button(f_box, text=self.T['btn_select_file'], command=self._select_file).pack(side=tk.LEFT, padx=5)
        t_box = ttk.Frame(file_group); t_box.pack(fill=tk.X, pady=5)
        ttk.Label(t_box, text=self.T['lbl_threshold']).pack(side=tk.LEFT)
        self.ts_limit = tk.IntVar(value=DEFAULT_MAX_TOKENS); ttk.Entry(t_box, textvariable=self.ts_limit, width=8).pack(side=tk.LEFT, padx=5)
        
        lang_group = ttk.LabelFrame(mid_frame, text=self.T['group_lang'], padding=10)
        lang_group.pack(side=tk.LEFT, fill=tk.BOTH, expand=True, padx=(5, 0))
        ttk.Label(lang_group, text=self.T['col_lang'], font=("Arial", 9, "bold")).grid(row=0, column=0, sticky=tk.W)
        ttk.Label(lang_group, text=self.T['col_trans'], font=("Arial", 9, "bold")).grid(row=0, column=1)
        ttk.Label(lang_group, text=self.T['col_exe'], font=("Arial", 9, "bold")).grid(row=0, column=2)
        
        order = ['zh', 'en', 'ja', 'ko', 'bo', 'pali', 'es', 'ru', 'fr', 'de', 'hi', 'no']
        for i, key in enumerate(order):
            row = i + 1
            ttk.Label(lang_group, text=self.T.get(f'lang_{key}', key)).grid(row=row, column=0, sticky=tk.W, padx=5, pady=2)
            val_trans = False; val_exe = (key == self.ui_lang) 
            vt = tk.BooleanVar(value=val_trans); self.vars_trans[key] = vt
            ve = tk.BooleanVar(value=val_exe); self.vars_exe[key] = ve
            ttk.Checkbutton(lang_group, variable=vt).grid(row=row, column=1)
            ttk.Checkbutton(lang_group, variable=ve).grid(row=row, column=2)

        main_content = ttk.PanedWindow(self.root, orient=tk.VERTICAL); main_content.pack(fill=tk.BOTH, expand=True, padx=10, pady=(0, 10))
        log_frame = ttk.LabelFrame(main_content, text=self.T['group_log']); main_content.add(log_frame, weight=1)
        self.log_text = scrolledtext.ScrolledText(log_frame, height=8, font=("Consolas", 9), state='normal'); self.log_text.pack(fill=tk.BOTH, expand=True)
        preview_frame = ttk.LabelFrame(main_content, text=self.T['group_preview']); main_content.add(preview_frame, weight=4)
        self.preview_area = scrolledtext.ScrolledText(preview_frame, font=("微软雅黑", 11)); self.preview_area.pack(fill=tk.BOTH, expand=True)
        self.progress = ttk.Progressbar(self.root, mode='determinate'); self.progress.pack(fill=tk.X, padx=10, pady=(0, 5))

    def _select_file(self):
        f = filedialog.askopenfilename(filetypes=[("Text Files", "*.txt")])
        if f: self.file_path.set(f)
    def log(self, msg):
        self.log_text.insert(tk.END, f"[{datetime.now().strftime('%H:%M:%S')}] {msg}\n"); self.log_text.see(tk.END)
    def stop(self):
        if self.is_running: self.stop_event.set(); self.log(self.T['msg_stop'])

    def start(self):
        if not self.file_path.get(): messagebox.showerror("Error", self.T['msg_err_nofile']); return
        selected_tasks = []
        selected_langs = set()
        for lang in ['zh', 'en', 'ja', 'ko', 'bo', 'pali', 'es', 'ru', 'fr', 'de', 'hi', 'no']:
            if self.vars_trans[lang].get(): 
                selected_tasks.append({'lang': lang, 'type': 'trans'})
                selected_langs.add(lang)
            if self.vars_exe[lang].get(): 
                selected_tasks.append({'lang': lang, 'type': 'exe'})
                selected_langs.add(lang)
        if not selected_tasks: messagebox.showerror("Error", self.T['msg_err_notask']); return
        self.is_running = True; self.stop_event.clear(); self.btn_start.config(state=tk.DISABLED)
        threading.Thread(target=self._run_process, args=(selected_tasks, list(selected_langs)), daemon=True).start()

    def _run_process(self, tasks, active_langs):
        input_file = self.file_path.get()
        base_name = os.path.splitext(input_file)[0]
        try:
            self.log(self.T['msg_read_start'])
            with open(input_file, 'r', encoding='utf-8') as f: content = f.read()
            segments = self.processor.smart_segmentation(content, self.ts_limit.get())
            total_segs = len(segments)
            self.log(self.T['msg_seg_done'].format(total_segs))
            
            split_file_path = f"{base_name}_split.txt"
            with open(split_file_path, 'w', encoding='utf-8') as f:
                for seg in segments: f.write(f"【{seg['title']}】\n{seg['content']}\n\n{'='*40}\n\n")
            self.log(self.T['msg_split_created'].format(os.path.basename(split_file_path)))

            handles = {}
            for t in tasks:
                suffix = LangConfig.get_file_suffix(t['lang'], t['type'])
                out_path = base_name + suffix
                f = open(out_path, 'w', encoding='utf-8')
                f.write(f"=== {t['type'].upper()} ({t['lang'].upper()}) ===\nSource: {os.path.basename(input_file)}\n\n")
                handles[(t['lang'], t['type'])] = f
                self.log(self.T['msg_file_created'].format(os.path.basename(out_path)))
            
            for lang in active_langs:
                suffix = LangConfig.get_file_suffix(lang, 'combined')
                out_path = base_name + suffix
                f = open(out_path, 'w', encoding='utf-8')
                f.write(f"=== CONTRAST VERSION ({lang.upper()}) ===\nSource: {os.path.basename(input_file)}\n\n")
                handles[(lang, 'combined')] = f
                self.log(self.T['msg_file_created'].format(os.path.basename(out_path)))

            ai = AiEngine(self.api_url.get(), self.api_key.get())
            for i, seg in enumerate(segments):
                if self.stop_event.is_set(): break
                title, text = seg['title'], seg['content']
                self.log(self.T['msg_processing'].format(i+1, total_segs, title))
                segment_results = {lang: {'trans': None, 'exe': None} for lang in active_langs}

                for t in tasks:
                    if self.stop_event.is_set(): break
                    lang, type_ = t['lang'], t['type']
                    prompt = LangConfig.get_trans_prompt(lang) if type_ == 'trans' else LangConfig.get_exe_prompt(lang)
                    validator = LangConfig.validate_pali if lang == 'pali' else None
                    
                    lang_name = self.T.get(f'lang_{lang}', lang)
                    task_display = self.T['col_trans'] if type_ == 'trans' else self.T['col_exe']
                    self.log(self.T['msg_generating'].format(lang_name, task_display))
                    
                    result = ai.process(title, text, prompt, self.model_name.get(), validator)
                    handles[(lang, type_)].write(f"【{title}】\n{result}\n\n{'='*60}\n\n"); handles[(lang, type_)].flush()
                    segment_results[lang][type_] = result
                    
                    self.preview_area.delete(1.0, tk.END)
                    self.preview_area.insert(tk.END, f"--- {lang_name} [{type_}] ---\n{title}\n\n{result}")
                
                for lang in active_langs:
                    combined_f = handles[(lang, 'combined')]
                    combined_text = f"【{title}】\n\n--- Source ---\n{text}\n"
                    res_trans = segment_results[lang].get('trans')
                    res_exe = segment_results[lang].get('exe')
                    if res_trans: combined_text += f"\n--- Translation ---\n{res_trans}\n"
                    if res_exe: combined_text += f"\n--- Analysis ---\n{res_exe}\n"
                    combined_text += f"\n{'='*60}\n\n"
                    combined_f.write(combined_text); combined_f.flush()

                self.progress['value'] = ((i + 1) / total_segs) * 100
            
            self.log("DONE!")
            messagebox.showinfo(self.T['msg_done_title'], self.T['msg_done_body'])
        except Exception as e:
            self.log(self.T['err_fatal'].format(str(e))); messagebox.showerror("Error", str(e))
        finally:
            if 'handles' in locals():
                for f in handles.values(): f.close()
            self.is_running = False; self.btn_start.config(state=tk.NORMAL); self.progress['value'] = 0

if __name__ == "__main__":
    launcher = LanguageLauncher()
    selected_lang = launcher.run()
    if selected_lang:
        root = tk.Tk()
        app = ZenUniversalApp(root, ui_lang=selected_lang)
        root.mainloop()
